/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System;
using Inet.Viewer.Resources;
using System.Globalization;

namespace Inet.Viewer.Data
{
    /// <summary>
    /// This class represent a single prompt value. e.g a number, string, date ... prompt
    /// @author LarsF
    /// </summary>
    public class SinglePromptValue : PromptValue
    {
        private readonly string DoubleFormat = new string('#', 32) + "0." + new string('#', 32);
        /// <summary>
        /// the value of this prompt </summary>
        private object value;

        /// <summary>
        /// Construcs a SinglePromptValue of this type. I </summary>
        /// <param name="type"> value type. </param>
        public SinglePromptValue(int type)
            : base(type)
        {
        }

        /// <summary>
        /// Creates a SinglePromptValue of the given type. </summary>
        /// <param name="value"> The value of this prompt value. </param>
        /// <param name="description"> The descript of this prompt value. </param>
        /// <param name="type"> the value type of this prompt value. </param>
        public SinglePromptValue(object value, string description, int type)
            : base(description, type)
        {
            this.value = value;
        }

        /// <summary>
        /// Creates a SinglePromptValue of the given type. </summary>
        /// <param name="type"> The value type of this prompt value. </param>
        /// <param name="valueString"> A string, which represents the value of this prompt. </param>
        /// <exception cref="Exception"> If valueString is not parsable. </exception>
        internal SinglePromptValue(int type, string valueString)
            : base("", type)
        {
            ParseValue(valueString);
        }

        /// <summary>
        /// Returns the value of this prompt value. </summary>
        /// <returns> the value of this prompt value. </returns>
        public override object Value
        {
            get
            {
                return value;
            }
        }

        /// <summary>
        /// Initializes the prompt value with the default value of its type.
        /// </summary>
        internal override void InitValue()
        {
            switch (this.Type)
            {
                case PromptData.Number:
                case PromptData.Currency:
                    this.value = new double?(0.0);
                    break;

                case PromptData.Boolean:
                    this.value = false;
                    break;
                case PromptData.String:
                    this.value = "";
                    break;
                case PromptData.Binary:
                    break;
                case PromptData.Time:
                    DateTime now = DateTime.Now;
                    this.value = new TimeSpan(now.Hour,now.Minute,now.Second);
                    break;
                case PromptData.Date:
                case PromptData.Datetime:
                    this.value = DateTime.Now;
                    break;
            }
        }

        /// <summary>
        /// Create the value of for this prompt value parsing the given string. </summary>
        /// <param name="valueString"> The string to parse. </param>
        /// <exception cref="Exception"> Throws an Exeption if the given string is not parsable. </exception>
        internal virtual void ParseValue(string valueString)
        {
            if (Type != PromptData.String && (valueString == null || valueString.Length == 0))
            {
                this.value = null;
                return;
            }
            string type = "";
            try
            {
                switch (this.Type)
                {
                    case PromptData.Number:
                    case PromptData.Currency:
                        type = strings.Prompt_TypeNumber;
                        if (valueString.IndexOf(CultureInfo.CurrentCulture.NumberFormat.CurrencyDecimalSeparator) == -1)
                        {
                            valueString = valueString.Replace(CultureInfo.InvariantCulture.NumberFormat.CurrencyDecimalSeparator, CultureInfo.CurrentCulture.NumberFormat.CurrencyDecimalSeparator);                            
                        }
                        this.value = double.Parse(valueString, CultureInfo.CurrentCulture);
                        break;
                    case PromptData.Boolean:
                        type = strings.Prompt_TypeBoolean;
                        this.value = bool.Parse(valueString);
                        break;
                    case PromptData.String:
                        type = strings.Prompt_TypeString;
                        this.value = valueString;
                        break;
                    case PromptData.Binary:
                        type = strings.Prompt_TypeBinary;
                        break;
                    case PromptData.Date:
                        type = strings.Prompt_TypeDate;
                        this.value = DateTime.Parse(valueString, CultureInfo.CurrentCulture).Date;
                        break;
                    case PromptData.Time:
                        type = strings.Prompt_TypeTime;
                        this.value = TimeSpan.Parse(valueString);
                        break;
                    case PromptData.Datetime:
                        type = strings.Prompt_TypeDatetime;
                        this.value = DateTime.Parse(valueString, CultureInfo.CurrentCulture);
                        break;
                }
            }
            catch (Exception e)
            {
                throw new ViewerException(strings.Prompt_InvalidValue + ". " + string.Format(strings.Prompt_EexpectedType, type), e);
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        internal override string ValueString
        {
            get
            {
                if (value == null)
                {
                    return '<'+strings.Prompt_None_Long+'>';
                }
                switch (Type % PromptData.Range)
                {
                    case PromptData.Currency:
                    case PromptData.Number:
                        return ((Double)value).ToString(DoubleFormat, CultureInfo.CurrentCulture);
                    case PromptData.Date:
                        return ((DateTime)value).ToShortDateString();
                    case PromptData.Datetime:
                        return ((DateTime)value).ToString(CultureInfo.CurrentCulture);
                    case PromptData.Time:
                        return value.ToString();
                    case PromptData.Boolean:
                        return ((bool)value) ? strings.Prompt_True : strings.Prompt_False;
                    default:
                        return value.ToString();
                }
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override string StringRepresentation
        {
            get
            {
                if (value == null)
                {
                    return "";
                }
                string p = "";
                switch (Type)
                {
                    case PromptData.Binary:
                        return Convert.ToString(this.value);
                    case PromptData.Date:

                        DateTime d = ((DateTime)value);
                        p = "Date(" + d.Year + "," + d.Month + "," + d.Day + ")";
                        break;
                    case PromptData.Datetime:
                        DateTime c = ((DateTime)value);
                        p = "DateTime(" + c.Year + "," + c.Month + "," + c.Day + "," + c.Hour + "," + c.Minute + "," + c.Second + ")";
                        break;
                    case PromptData.Time:
                        TimeSpan time = ((TimeSpan)value);
                        p = "Time(" + time.Hours + "," + time.Minutes + "," + time.Seconds + ")";
                        break;
                    case PromptData.Boolean:
                        p = value.ToString();
                        break;
                    case PromptData.Currency:
                    case PromptData.Number:
                        double number2 = (double)value;
                        if ((double)number2 == double.PositiveInfinity)
                        {
                            p = "'<'";
                        }
                        else if ((double)number2 == double.NegativeInfinity)
                        {
                            p = "'>'";
                        }
                        else
                        {
                            p = number2.ToString(CultureInfo.InvariantCulture);
                        }
                        break;
                    case PromptData.String:
                        string val = ValueString;
                        val = val.Replace("'", "''");
                        p = "'" + val + "'";
                        break;
                }
                return p;
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override bool Equals(object obj)
        {
            if (!(obj is SinglePromptValue))
            {
                return false;
            }
            SinglePromptValue v = (SinglePromptValue)obj;
            bool valuesAreEqual = false;
            if (v.value == null)
            {
                valuesAreEqual = value == null;
            }
            else
            {
                valuesAreEqual = v.value.Equals(value);
            }
            bool descriptionsAreEqual = false;
            string desc = v.Description;
            if (desc == null)
            {
                descriptionsAreEqual = Description == null;
            }
            else
            {
                descriptionsAreEqual = desc.Equals(Description);
            }
            return valuesAreEqual && descriptionsAreEqual;
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override int GetHashCode()
        {
            if (value == null)
            {
                return 0;
            }
            return value.GetHashCode();
        }
    }
}